function [bms,envs,instfs,delays]=gammatonebank(x,lowcf,highcf,numchans,fs,align)
% Implements a bank of fourth-order gammatone filters
%
%   [bms,envs,instfs,delays] = ...
%       gammatonebank(x,lowcf,highcf,numchans,fs,align)
%
%   This function takes an input vector and passes it
%   through a bank of fourth-order gammatone filters,
%   equally spaced on the ERB-rate scale.
%   
%   bms=gammatonebank(x,lowcf,highcf,numchans,fs) returns an
%   array of numchans gammatone filterbank outputs in
%   response to vector x (sampled at frequency fs). Centre
%   frequencies are equally spaced on the ERB-rate scale,
%   between lowcf and highcf.
% 
%   [bms,envs,instfs]=gammatonebank(...) optionally returns
%   the instantaneous envelopes envs and instantaneous
%   frequencies instfs for each filter.
% 
%   [...]=gammatonebank(...,align) allows phase alignment to
%   be applied. With align=false, no alignment is applied
%   (default). With align=true, fine structure and envelope
%   alignment is applied so that the impulse response peaks
%   occurs at t=0.
% 
%   [bms,envs,instfs,delays]=gammatonebank(...) returns the
%   delay (in samples) removed by the phase alignment of
%   each gammatone filter, i.e. delays(n)=0 if align=false.
%
%   Adapted from code by Guy Brown, University of Sheffield
%   and Martin Cooke.
% 
%   See also gammatone, MakeErbCFs.

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:01:41 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 71 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

if nargin<5
    error('Not enough input arguments')
end
if nargin<6
    align = false;
end
if numel(x)~=max(size(x))
    error('x must be a vector')
end
x = reshape(x,1,length(x));

bms = zeros(numchans,length(x));
envs = zeros(numchans,length(x));
instfs = zeros(numchans,length(x));
delays = zeros(numchans,1);

cfs=MakeErbCFs(lowcf,highcf,numchans);
for c=1:numchans
  cf=cfs(c);
  [bm,env,instf,delay]=gammatone(x,fs,cf,align);
  bms(c,:) = bm;
  envs(c,:) = env;
  instfs(c,:) = instf;
  delays(c) = delay;
end

% [EOF]
