function separate(hObject,handles)
% The main separation function.
% 
%   separate(hObject,handles)
% 
%   Refers to comments within the file for specific help on
%   each of its elements

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:58:52 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 287 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

d = cd; % capture current directory
cd(handles.directory) % CD to GUI directory

%% Initiation and User Variables
% This section gathers the user inputs from the GUI and defines some other
% variables.

% The files
target_files = get(handles.lst_targets,'string'); % Array of filenames for targets
interferer_files = get(handles.lst_interferers,'string'); % Array of filenames for interferers

% Mixture variables
Rooms = handles.availableRooms(handles.Rooms_to_test_IDs);
% The Target-to_Interferer Ratio (TIR)
user_TIR = eval(get(handles.txt_TIR,'string'));
% Azimuth of target impulse response (Room is chosen above)
user_azimuth_target = eval(get(handles.txt_azimuth_target,'string'));
% Azimuth of interferer impulse response (Room is chosen above)
user_azimuth_interferer = eval(get(handles.txt_azimuth_interferer,'string'));

% Other Variables
lowcf = str2double(get(handles.txt_lowcf,'string')); 
highcf = str2double(get(handles.txt_highcf,'string')); 
numchans = str2double(get(handles.txt_numchans,'string')); 
maxlag = str2double(get(handles.txt_maxlag,'string')); 
frame = str2double(get(handles.txt_frame,'string')); 
noverlap = str2double(get(handles.txt_noverlap,'string'));

fs = handles.fs;
frame2 = 20; % No of frames for rate threshold calculation
noverlap2 = 10; % Frame overlap for rate threshold calculation
rate_threshold = -11; % Energy Rate Threshold [dB]
theta_c_dB = -160; % CCG Azimuth extraction interferer floor [dB]
b = 15; % Spectral Energy Normalisation Factor
sd_min = 0.75; % sccg standard deviation min
sd_max = 4.5; % sccg standard deviation max
sd = linspace(sd_max,sd_min,numchans);
theta_IBM = 0; % dB % IBM threshold
maxdeg = 90; % degrees
dB_SPL = 60;

frame_d = (frame/1000)*fs;
maxlag_d = maxlag*(fs/1000);
rad = ((-maxdeg:maxdeg)/180)*pi;

% precedence parameter
prec_params_1 = eval(get(handles.txtPrecedenceParameter1,'string'));
prec_params_2 = eval(get(handles.txtPrecedenceParameter2,'string'));

% Anechoic room for ILD template
anechoic_rooms = get(handles.cbo_anechoic_room,'string');
value = get(handles.cbo_anechoic_room,'value');
anechoic_room = anechoic_rooms{value};

%% Define operation for chosen precedence model
% This section sets up the separation to run with the chosen precedence
% model, i.e. by setting the precedence function handle, model name and
% ensuring any additional mex functions are compiled for the system.

available_models = get(handles.cboPrecedenceModel,'string');
chosen_model = get(handles.cboPrecedenceModel,'value');
displayname = available_models{chosen_model};

modelname = get_model_parameter(handles.config,displayname,'modelname');
prec_fhandle = eval(get_model_parameter(handles.config,displayname,'prec_fhandle'));

% Check warp function is compiled
check_mex_compiled('-largeArrayDims','warp_ccg_c.c')

%% Define operation for chosen Inner Hair Cell model
% This section sets up the separation to run with the chosen IHC model by
% setting the IHC function handle.

IHC_models = get(handles.cboIHCmodel,'string');
chosen_IHC = get(handles.cboIHCmodel,'value');
switch IHC_models{chosen_IHC}
    case 'Half-wave Rectify'
        ihc_fhandle = @ihc_hwr;
    case 'Meddis'
        ihc_fhandle = @ihc_meddis;
    case 'Faller & Merimaa'
        ihc_fhandle = @ihc_fallermerimaa;
    case 'Bypass'
        ihc_fhandle = @ihc_bypass;
end

%% Check results file
% This section performs several checks on the results file. Firstly, it
% checks if it already exists. If it does then it checks for results
% calculated with the chosen precedence model (offering the user the
% options to append or overwrite if they do). If results don't exist then
% it creates the necessary structure to store the data.

[results_path,results_filename] = fileparts(get(handles.txt_results_file,'string'));

if ~isdir(results_path)
    mkdir(results_path);
end

% Build the revision profile for the source code.
% If results are being added to a pre-existing results file, a check is
% performed below to ensure that the source code hasn't changed 
revision_profile = build_svn_profile({handles.directory,...
    [handles.directory filesep 'Functions'],...
    [handles.directory filesep 'Library']},...
    {'Date','Revision'},{'*.m','*.c'});

if exist([results_path filesep results_filename '.mat'],'file') == 2
    load([results_path filesep results_filename '.mat'])
    % check the revision profile for data, warning the user if it has changed or missing.
    err = check_revision_profile(results,revision_profile,'revision_profile'); %#ok<NODEF>
    displaynames = cellfun(@(x) x.displayname,struct2cell(results),'uni',false);
    if any(err)
        w = warndlg([{horzcat('Differences in the revision profiles stored in the results, ',...
        'compared to the current source code, were detected. This suggests that there may be ',...
        'differences between versions used to generate the results. In order to ensure ',...
        'data integrity and eliminate any possible source of error, either remove or overwrite ',...
        'the results for models that are different.');
            '';
            'Differences were detected in:';
            ''};
            sort(displaynames(err))],...
            'Warning!');
        uiwait(w)
    end
    if isfield(results,modelname)
        cont_box = questdlg('Data for this model already exists. Do you wish to overwrite or append the data?',...
            'Results already exist!','Overwrite','Append','Cancel','Overwrite');
        switch cont_box
            case'Overwrite'
                results.(modelname).revision_profile = revision_profile;
                counter = 0;
            case 'Append'
                counter = length(results.(modelname).TIR);
            otherwise
                return
        end
    else
        results.(modelname).revision_profile = revision_profile;
        counter = 0;
    end
else
    results = struct(modelname, struct);
    results.(modelname).revision_profile = revision_profile;
    counter = 0;
end

%% Create results fields

% write precedence parameter name to results
results.(modelname).prec_par_1_name = get_model_parameter(handles.config,displayname,'prec_par_1_label');
results.(modelname).prec_par_2_name = get_model_parameter(handles.config,displayname,'prec_par_2_label');

% arrange results structure
results.(modelname).displayname = displayname;

iterations = length(Rooms)*length(target_files)*length(interferer_files)*length(user_TIR)*length(user_azimuth_target)*length(prec_params_1)*length(prec_params_2);
iteration_counter = 0;

%% ITD-Azimuth warping function
% Derives mappings for azimuth based on ITD

cfs = round(MakeErbCFs(lowcf, highcf, numchans));  % Create filterbank centre frequencies

warpers = get_warp_func(modelname,ihc_fhandle,cfs,handles.BRIR_files,anechoic_room,fs,maxlag,maxdeg,handles.directory);

%% ILD Template
% Derives mappings for azimuth based on ILD

ild_f = find(cfs>2800); % Choose only frequency channels above 2.8 kHz

[ild_template,ild_angles] = get_ild_template(handles.BRIR_files,anechoic_room,cfs(ild_f),dB_SPL,handles.directory);

noise_floor = (dB2num(0)^3.333)^2; % 0 dB SPL based on Meddis' numeric range

%% Check user-specified stimuli
% Check the stimuli chosen by the user (forces them to have a common
% length, sampling frequency and to be mono)

the_target_wavs = check_stimuli(target_files,fs);
the_interferer_wavs = check_stimuli(interferer_files,fs);
difflength = size(the_target_wavs,1)-size(the_interferer_wavs,1);
if difflength>0
    the_interferer_wavs = [the_interferer_wavs; zeros(difflength,size(the_interferer_wavs,2))];
elseif difflength<0
    the_target_wavs = [the_target_wavs; zeros(-difflength,size(the_target_wavs,2))];
end

%% Begin looped calculations
% begin calculations for each experimental variable: precedence parameter,
% stimuli reverb times, TIR and azimuthal separation

hwb = waitbar(0,'Now processing...','name','Processing...');

for prec_par_1 = prec_params_1
for prec_par_2 = prec_params_2
for target = 1:length(target_files)
for interferer = 1:length(interferer_files)
for the_room = 1:length(Rooms)
for tir = 1:length(user_TIR)
for the_azimuth = 1:length(user_azimuth_target)

% Provide user feedback on progress
    
azimuth_t = user_azimuth_target(the_azimuth);
azimuth_n = user_azimuth_interferer(the_azimuth);
Room = Rooms{the_room};
TIR = user_TIR(tir);

if ishandle(hwb)
    waitbar(iteration_counter/iterations,hwb,['Now processing ' num2str(iteration_counter+1) ' of ' num2str(iterations)])
else
    hwb = waitbar(iteration_counter/iterations,...
        ['Now processing ' num2str(iteration_counter+1) ' of ' num2str(iterations)],'name','Processing...');
end
counter = counter+1;
iteration_counter = iteration_counter + 1;

disp([num2str(iteration_counter) ' of ' num2str(iterations)])

%% Calculate mixture
% Create the cocktail party mixture and provide stimuli. Derive the number
% of frames.

% Extract interferer file name from path
[~,interferer_filename] = fileparts(interferer_files{interferer});
[~,target_filename] = fileparts(target_files{target});

[mixture,target_wav,interferer_wav] = create_mixture(handles.BRIR_files,the_target_wavs(:,target),the_interferer_wavs(:,interferer),Room,...
    azimuth_t,azimuth_n,TIR,fs);

mixture = mixture';

% Create and convert some variables
frameCount = floor((length(mixture)-maxlag_d-1)/(frame_d));
frameCount = frameCount-noverlap+1;

%% Gammatone Filterbank
% Filter the mixture and set to numeric RMS level corresponding to dB_SPL

start1 = tic;

% filter input signal through a bank of gammatone filters
[bm_L,env_L] = gammatonebank(mixture(1,:), lowcf, highcf, numchans, fs);
[bm_R,env_R] = gammatonebank(mixture(2,:), lowcf, highcf, numchans, fs);

amp = dB2num(dB_SPL)/max([calc_rms(bm_L,2); calc_rms(bm_R,2)]);

bm_L = amp.*bm_L;
bm_R = amp.*bm_R;
env_L = amp.*env_L;
env_R = amp.*env_R;

%% IHC Processing
% Perform Inner Hair Cell processing

hc_L = ihc_fhandle(bm_L,env_L,fs);
hc_R = ihc_fhandle(bm_R,env_R,fs);

%% Calculate clean target and interferer signal energy
% For calculating the ideal mask

% target
[~,envc] = gammatonebank(target_wav', lowcf, highcf, numchans, fs);
anfr_clean = calc_anfr(envc,fs,frame_d,frameCount);
eng_LR_target = (anfr_clean.^3.333).^2;

% interferer
[~,envn] = gammatonebank(interferer_wav', lowcf, highcf, numchans, fs);
anfr_int = calc_anfr(envn,fs,frame_d,frameCount);
eng_LR_interferer = (anfr_int.^3.333).^2;

%% Auditory Nerve Firing Rate

anfr_L = calc_anfr(env_L,fs,frame_d,frameCount);
anfr_R = calc_anfr(env_R,fs,frame_d,frameCount);

anfr_LR = (0.5.*((anfr_L).^3.333) + 0.5.*((anfr_R).^3.333)).^0.3;

%% Localisation
% Calculate cross-correlograms and perform precedence processing. Assign
% sources to azimuths.
start2 = tic;
ccg = prec_fhandle(hc_L,hc_R,env_L,env_R,fs,prec_par_1,prec_par_2,maxlag_d,frame_d,frameCount,numchans,noverlap,Room);
prec_time = toc(start2);

% ccg should have dimensions [lag,channel,frame]
[ccg_azi,sccg] = warp_ccg(ccg,sd,warpers,numchans);

theta_c = max(ccg_azi(:))*(10^(theta_c_dB/10)); % Cross-correlogram grouping threshold

psccg = mean(mean(sccg,3),2);
psccg = (1/max(psccg)).*psccg;
% figure; plot(-90:90,psccg); set(gca,'xlim',[-90 90],'xtick',-90:30:90)

% Extract peaks in PSCCG
azimuth = zeros(size(psccg));
psccg_peaks = localpeaks(psccg);
azimuth(psccg_peaks) = psccg(psccg_peaks);

% Find two largest cross-correlations (assume these are target and interferer sounds)
azi_1 = find(azimuth==max(azimuth));
azimuth(azi_1) = 0;
azi_2 = find(azimuth==max(azimuth));
if length(azi_2)>1 % if only one peak is found
    azi_2 = (2*maxdeg)+2-azi_1;
    if azi_2 == azi_1
        azi_2 = (2*maxdeg)+1;
    end
end

% Assign target and intrusion azimuths
if azimuth_t < azimuth_n
    target_index = min(azi_1,azi_2);
    interferer_index = max(azi_1,azi_2);
else
    interferer_index = min(azi_1,azi_2);
    target_index = max(azi_1,azi_2);
end

% specify azimuths in degrees
target_azimuth = (rad(target_index)/pi)*180;
interferer_azimuth = (rad(interferer_index)/pi)*180; 
disp(['Azimuths: ' num2str(target_azimuth) ', ' num2str(interferer_azimuth)])

%% ILD
% Calculate ILDs for stimulus

ild = zeros(length(ild_f),length(anfr_L));

% Calculate the energy
eng_L = (anfr_L(ild_f,:).^3.333).^2; 
eng_L = eng_L + noise_floor;

eng_R = (anfr_R(ild_f,:).^3.333).^2;
eng_R = eng_R + noise_floor;

for i = 1:length(ild_f)
    for j = 1:frameCount
        ild(i,j) = 10.*log10(eng_L(i,j)/eng_R(i,j));
    end
end

%% Grouping by common azimuth
% Estimates target and interferer energy according to cross-correlation
% strength at chosen azimuths

m = (squeeze(ccg_azi(target_index,:,:)) > squeeze(ccg_azi(interferer_index,:,:)))...
    & (squeeze(ccg_azi(target_index,:,:)) > theta_c);

m = +m;

%% ILD Constraint
% Remove units where the ILD doesn't get close to the template for the head

azi_intervals = sort(handles.BRIR_files.(['Room_' anechoic_room]).angles);

% choose the azimuth templates closest to the obtained azimuths:
v = abs(azi_intervals-target_azimuth);
azi_choice = ild_angles==azi_intervals(v==min(v));

%%%%
for j = 1:frameCount;
    if abs(ild(:,j) - ild_template(:,azi_choice)) > 1
            m(ild_f,j) = 0;
    end
end

%% Rate Threshold
% Remove units according to an energy rate threshold

eng_LR = (anfr_LR.^3.333).^2;

e = zeros(size(eng_LR));

for j = 1:noverlap2:frameCount-frame2
    for i = 1:numchans
        e(i,j:j+frame2) = sum(eng_LR(i,j:j+frame2))/frame2;
    end
end

m(10*log10(eng_LR./e) < rate_threshold) = 0;

%% Spectral Energy Normalisation
% Normalise the spectral energy using only information from target units

eta = zeros(numchans,1);

for i = 1:numchans
    units = find(m(i,:)==1);
    l = min(length(units),round(frameCount/b));
    Gamma = sort(anfr_LR(i,units));
    Gamma = Gamma(end-l+1:end);
    eta(i) = mean(Gamma);
end

% if there are no target units in a channel then eta will be NaN. This
% value is irrelevant as the channel will be discarded during the
% re-synthesis. The corresponding normalisation factor will be set to 0.

%% Ideal Binary Mask

m_IBM = zeros(numchans,frameCount);
m_IBM(10*log10(eng_LR_target./eng_LR_interferer) > theta_IBM) = 1;

%% Re-synthesis
% Resythesise the target from the calculated and ideal masks

output_L = resynthesise(mixture(1,:),fs,cfs,m,frame_d,eta);
output_R = resynthesise(mixture(2,:),fs,cfs,m,frame_d,eta);
output_L_IBM = resynthesise(mixture(1,:),fs,cfs,m_IBM,frame_d,eta);
output_R_IBM = resynthesise(mixture(2,:),fs,cfs,m_IBM,frame_d,eta);

if get(handles.chkPlay,'value')==1 || get(handles.chkSave,'value')==1
    output = [output_L' output_R'];  output = output./max(abs(output(:)));
    if get(handles.chkPlay,'value')==1
        sound(output,fs,16);
    end
    if get(handles.chkSave,'value')==1
        [~,target_filename] = fileparts(target_files{target});
        wavwrite(0.5.*output,fs,16,[results_path filesep modelname '_' num2str(prec_par_1) '_' num2str(prec_par_2) '_' target_filename '_' interferer_filename '_' Room '_' num2str(TIR) '_' num2str(azimuth_t) '_' num2str(azimuth_n) '.wav']);
    end
end

overall_time = toc(start1);

disp(['Processing time: ' num2str(overall_time) ' (overall), ' num2str(prec_time) ' (precedence model)'])

%% Evaluation
% Evaluate the processing: SNR, SINR and IBMR

ibmr = calc_ibmr(m,m_IBM);
disp(['IBMR: ' num2str(ibmr)])
disp(' ')

SINR_L = calc_snr(output_L,output_L_IBM);
SINR_R = calc_snr(output_R,output_R_IBM);

s(1,:) = target_wav(1:length(output_L));
s(2,:) = interferer_wav(1:length(output_L));

SNR_L = calc_snr(output_L,s(1,:));
SNR_R = calc_snr(output_R,s(1,:));

SNR_L_IBM = calc_snr(output_L_IBM,s(1,:));
SNR_R_IBM = calc_snr(output_R_IBM,s(1,:));

%% Write results to file

% build results structure
results.(modelname).target{counter} = target_filename;
results.(modelname).interferer{counter} = interferer_filename;
results.(modelname).Room{counter} = Room;
results.(modelname).TIR(counter) = TIR;
results.(modelname).azi_t(counter) = azimuth_t;
results.(modelname).azi_i(counter) = azimuth_n;
results.(modelname).prec_par_1(counter) = prec_par_1;
results.(modelname).prec_par_2(counter) = prec_par_2;
%
results.(modelname).SINR_L(counter) = SINR_L;
results.(modelname).SINR_R(counter) = SINR_R;
results.(modelname).SNR_L(counter) = SNR_L;
results.(modelname).SNR_R(counter) = SNR_R;
results.(modelname).IBMR(counter) = ibmr;
results.(modelname).SNR_L_IBM(counter) = SNR_L_IBM;
results.(modelname).SNR_R_IBM(counter) = SNR_R_IBM;
%
results.(modelname).prec_time(counter) = prec_time;
results.(modelname).overall_time(counter) = overall_time;

save([results_path filesep results_filename], 'results')

end
end
end
end
end
end
end

if ishandle(hwb)
    close(hwb)
end

msgbox({'Data saved to:'; [results_path filesep results_filename '.mat']}, 'Processing complete','help','modal')
disp(['Processing complete. Data saved to: ' results_path filesep results_filename '.mat'])
guidata(hObject,handles)

cd(d)

% [EOF]
