function [mixture,target_wav,interferer_wav] = create_mixture(BRIR_files,the_target_wav,the_interferer_wav,room,azimuth_t,azimuth_n,TIR,fs)
% Create the cocktail party mixture.
% 
%   [mixture,target_wav,interferer_wav] = ...
%       create_mixture(BRIR_files,the_target_wav,...
%       the_interferer_wav,room,azimuth_t,azimuth_n,TIR,fs)
% 
%   [mixture,target_wav,interferer_wav]=create_mixture(...)
%   creates a cocktail party mixture containing 2 sound
%   sources using the BRIR_files filename database and room
%   tag. The target wav file the_target_wav is placed at
%   azimuth azimuth_t and the the interferer
%   the_interferer_wav is placed at azimuth azimuth_n. The
%   RMS levels of the wav files are set to differ by TIR dB
%   and these modified files are written to target_wav and
%   interferer_wav. The outputs have the sampling frequency
%   fs Hz.
% 
%   See also CHECK_STIMULI.

% !---
% ==========================================================
% Last changed:     $Date: 2011-09-13 17:02:31 +0100 (Tue, 13 Sep 2011) $
% Last committed:   $Revision: 285 $
% Last changed by:  $Author: mu31ch $
% ==========================================================
% !---

target_IR_filename = get_IR_filename(BRIR_files,room,azimuth_t);
interferer_IR_filename = get_IR_filename(BRIR_files,room,azimuth_n);

% get mixture parameters

[target_IR_wav,target_IR_fs] = wavread(target_IR_filename);

[interferer_IR_wav,interferer_IR_fs] = wavread(interferer_IR_filename);

if target_IR_fs~=fs
    error(['The target BRIR (Room = ' room  ', azimuth = ' target_azi ') sampling frequency does not match the master sampling frequency'])
end

if interferer_IR_fs~=fs
    error(['The interferer BRIR (RT=' room  ', azimuth=' target_azi ') sampling frequency does not match the master sampling frequency'])
end

% create mixture

TIR2 = 10^(TIR/20);

rms_x = calc_rms(the_target_wav);
rms_y = calc_rms(the_interferer_wav);
target_wav = the_target_wav;
the_interferer_wav = (rms_x/rms_y).*the_interferer_wav; % normalise the relative rms levels (yep, quick and dirty!)
interferer_wav = the_interferer_wav/TIR2;

target = [conv_fft(target_wav,target_IR_wav(:,1)) conv_fft(target_wav,target_IR_wav(:,2))];
target = crop(target,target_wav);

interferer = [conv_fft(interferer_wav,interferer_IR_wav(:,1)) conv_fft(interferer_wav,interferer_IR_wav(:,2))];
interferer = crop(interferer,interferer_wav);

% Normalise signals
G = max([interferer\interferer_wav; target\target_wav]);
interferer = interferer.*G;
target = target.*G;

mix_length = min(max(size(target)),max(size(interferer)));
mixture = target(1:mix_length,:) + interferer(1:mix_length,:);

function y = crop(x,source)

% crop convolved signals and remove delay due to convolution

cc(:,1) = xcorr(abs(x(1:length(source),1)),abs(source));
cc(:,2) = xcorr(abs(x(1:length(source),2)),abs(source));
lags = -length(source):length(source);
delay = max(2,min(lags(cc(:,1)==max(cc(:,1))),lags(cc(:,2)==max(cc(:,2)))));
y = x(delay-1:end,:);
if length(y)>length(source)
    y = y(1:length(source),:);
elseif length(y)<length(source)
    y = [y; zeros(length(source)-length(y),2)];
end

% [EOF]
